# src/pjots_zen_tools/hub/registry.py
from __future__ import annotations

from typing import Dict, Callable, Any, Optional
import threading
import json
import os

__all__ = [
    "register_tool",
    "unregister_tool",
    "update_tool",
    "set_installed",
    "is_registered",
    "get_tools",
    "clear_registry",
    "pin_version",
    "get_pinned_version",
    "save_pinning_state",
    "load_pinning_state",
]

# Internal storage
_REGISTRY: Dict[str, Dict[str, Any]] = {}
_PINNED: Dict[str, str] = {}
_LOCK = threading.RLock()


def _normalize_meta(
    module_id: str,
    title: str,
    launcher: Optional[Callable],
    icon: str,
    version: str,
    tags: Optional[list],
    description: str,
    path: Optional[str] = None,
    installed: Optional[bool] = None,
) -> Dict[str, Any]:
    """
    Normalizes metadata. 'installed' defaults to True for registered tools.
    """
    return {
        "module_id": module_id,
        "title": title or module_id,
        "launcher": launcher,
        "icon": icon or "",
        "version": version or "0.0.0",
        "tags": list(tags or []),
        "description": description or "",
        # optional fields that, among others, the Hub uses/passes on:
        "path": path or "",
        "installed": True if installed is None else bool(installed),
    }


def register_tool(
    module_id: str,              # e.g. "vertex_distributor"
    title: str,
    launcher: Callable | None,
    icon: str = "",
    version: str = "1.0.0",
    tags: Optional[list] = None,
    description: str = "",
    *,
    path: Optional[str] = None,
    installed: Optional[bool] = None,
) -> None:
    """
    Registers/updates a tool in the in-memory catalog.
    Backwards-compat: same signature as before plus optional keywords.
    """
    meta = _normalize_meta(
        module_id=module_id,
        title=title,
        launcher=launcher,
        icon=icon,
        version=version,
        tags=tags,
        description=description,
        path=path,
        installed=installed,
    )
    with _LOCK:
        _REGISTRY[module_id] = meta


def unregister_tool(module_id: str) -> bool:
    """
    Completely removes a tool from the registry.
    Returns: True if it existed and was removed.
    """
    with _LOCK:
        return _REGISTRY.pop(module_id, None) is not None


def set_installed(module_id: str, installed: bool) -> bool:
    """
    Marks an already registered tool as installed/not installed,
    without losing metadata. Practical for immediate UI feedback after uninstall.
    Returns: True on success.
    """
    with _LOCK:
        meta = _REGISTRY.get(module_id)
        if not meta:
            return False
        meta["installed"] = bool(installed)
        # Optionally disable launcher if not installed (stronger protection)
        if not installed:
            meta["launcher"] = None
        return True


def update_tool(module_id: str, **fields: Any) -> bool:
    """
    Updates individual fields of a registered tool.
    Allows 'version', 'icon', 'launcher', 'tags', 'description', 'path', 'installed', among others.
    Returns: True on success.
    """
    with _LOCK:
        meta = _REGISTRY.get(module_id)
        if not meta:
            return False
        # Only allow known keys (sanitizing)
        allowed = {
            "title", "launcher", "icon", "version", "tags",
            "description", "path", "installed",
        }
        for k, v in list(fields.items()):
            if k not in allowed:
                continue
            if k == "tags" and v is not None:
                meta[k] = list(v)
            elif k == "installed":
                meta[k] = bool(v)
                if not meta[k]:
                    meta["launcher"] = None
            else:
                meta[k] = v
        return True


def is_registered(module_id: str) -> bool:
    with _LOCK:
        return module_id in _REGISTRY


def get_tools(*, copy: bool = True) -> Dict[str, Dict[str, Any]]:
    """
    Returns the registry. Default: copy (avoid side effects).
    For performance-critical call sites, copy=False can be used.
    """
    with _LOCK:
        return dict(_REGISTRY) if copy else _REGISTRY


def clear_registry() -> None:
    """Clears the registry (for tests/reload)."""
    with _LOCK:
        _REGISTRY.clear()


# ---------------------------------------------------------------------
# Version Pinning Functions
# ---------------------------------------------------------------------

def _get_pinning_file_path() -> str:
    """Get path for storing pinning state."""
    try:
        # Try Maya prefs dir first
        import maya.cmds as cmds
        prefs_dir = cmds.internalVar(userPrefDir=True).replace("\\", "/")
    except Exception:
        # Fallback to home directory
        prefs_dir = os.path.join(os.path.expanduser("~"), ".pjots_zen").replace("\\", "/")

    return os.path.join(prefs_dir, "version_pinning.json").replace("\\", "/")


def pin_version(module_id: str, version: str) -> bool:
    """
    Pin a tool to a specific version (or "newest" for auto-updates).
    Works even if the tool is not installed (i.e. not in _REGISTRY).
    Returns True on success.
    """
    try:
        ver = str(version or "newest").strip() or "newest"
    except Exception:
        ver = "newest"

    with _LOCK:
        # Store in global pin map (works for non-installed tools)
        if ver == "newest":
            _PINNED.pop(module_id, None)
        else:
            _PINNED[module_id] = ver

        # Also reflect into installed registry meta (if present)
        meta = _REGISTRY.get(module_id)
        if meta is not None:
            meta["pinned_version"] = ver

    return save_pinning_state()


def get_pinned_version(module_id: str) -> str:
    """
    Get the pinned version for a tool.
    Returns "newest" if not pinned.
    """
    with _LOCK:
        # If installed meta has it, use it
        meta = _REGISTRY.get(module_id)
        if meta and "pinned_version" in meta:
            return str(meta.get("pinned_version", "newest"))
        # Fallback for non-installed tools
        return str(_PINNED.get(module_id, "newest"))


def save_pinning_state() -> bool:
    """
    Save pinned versions to persistent storage.
    Only stores entries where pinned_version != "newest".
    Returns True on success.
    """
    try:
        with _LOCK:
            pinned_states = {
                mid: ver for mid, ver in _PINNED.items()
                if ver and str(ver) != "newest"
            }

        file_path = _get_pinning_file_path()
        os.makedirs(os.path.dirname(file_path), exist_ok=True)

        with open(file_path, "w", encoding="utf-8") as f:
            json.dump(pinned_states, f, indent=2, ensure_ascii=False)

        return True
    except Exception:
        return False


def load_pinning_state() -> bool:
    """
    Load pinned versions from persistent storage and apply to:
    - _PINNED (always)
    - _REGISTRY meta (if installed)
    Returns True on success.
    """
    try:
        file_path = _get_pinning_file_path()
        if not os.path.exists(file_path):
            # No file is fine; just keep empty pins
            with _LOCK:
                _PINNED.clear()
            return True

        with open(file_path, "r", encoding="utf-8") as f:
            pinned_states = json.load(f)

        if not isinstance(pinned_states, dict):
            return False

        with _LOCK:
            _PINNED.clear()
            for module_id, pinned_version in pinned_states.items():
                ver = str(pinned_version or "newest")
                if ver != "newest":
                    _PINNED[str(module_id)] = ver

            # Apply to installed tools too
            for module_id, ver in _PINNED.items():
                if module_id in _REGISTRY:
                    _REGISTRY[module_id]["pinned_version"] = ver

        return True
    except Exception:
        return False
