# -*- coding: utf-8 -*-
"""
pjots_zen_tools.core.shelves (v3)
Robuste Shelf-Utilities für Maya 2026+.

Funktionen:
- ensure_wrapper_file(cmds=None, name="PjotsZenTools", bootstrap=("hub","vd")) -> str
    Schreibt eine MINIMALE shelf_<name>.mel, die den Python-Builder startet.
    Idempotent; erzeugt niemals *.mel.mel.

- save_shelf(cmds=None, name="PjotsZenTools") -> str
    Kompatibler Alias für ensure_wrapper_file (Persistenz via Wrapper).

- build_or_sync(cmds=None, name="PjotsZenTools", bootstrap=None) -> None
    Baut (oder aktualisiert) das Shelf in der UI und räumt auf:
      1) Shelf-Layout sicherstellen
      2) Optional Bootstrap-Buttons anlegen (Hub/VD)
      3) Registry lesen und verwaiste zen:* Buttons entfernen (Bootstrap geschützt)
      4) Wrapper-Datei sicherstellen (Persistenz)

- remove_module_button(cmds, name, module_id) -> int
    Entfernt Buttons mit ann == "zen:<module_id>".

- sync_shelf_with_registry(cmds, name, registry) -> int
    Entfernt verwaiste zen:* Buttons, die nicht mehr in der Registry stehen.

Hinweise:
- "name" wird normalisiert (ohne 'shelf_' und ohne '.mel').
- Persistenz: KEIN mel.saveShelf / saveAllShelves -> nur der Wrapper.
- Buttons werden über die Annotation "zen:<module_id>" eindeutig identifiziert.
"""

from __future__ import annotations

import os
from typing import Dict, Iterable, List, Optional, Tuple

try:
    from pjots_zen_tools import resources as R
except Exception:
    R = None  # optional


# -----------------------------------------------------------------------------
# Normalisierung & Pfade
# -----------------------------------------------------------------------------
def _normalize_name(name: str) -> str:
    """Entferne führendes 'shelf_' und Endung '.mel' aus einem Shelf-Namen."""
    n = (name or "").strip()
    if n.lower().startswith("shelf_"):
        n = n[6:]
    if n.lower().endswith(".mel"):
        n = n[:-4]
    return n


def _ensure_cmds(cmds):
    """Sorge dafür, dass maya.cmds verfügbar ist."""
    if cmds is None:
        import maya.cmds as _cmds  # type: ignore
        return _cmds
    return cmds


def _user_prefs_dir(cmds) -> str:
    return cmds.internalVar(userPrefDir=True)


def _shelves_dir(cmds) -> str:
    return os.path.join(_user_prefs_dir(cmds), "shelves")


def _shelf_file_path(cmds, name: str) -> str:
    """Pfad zur Wrapper-Datei shelf_<name>.mel."""
    n = _normalize_name(name)
    return os.path.join(_shelves_dir(cmds), f"shelf_{n}.mel")


# -----------------------------------------------------------------------------
# Shelf-Layout-Utilities
# -----------------------------------------------------------------------------
def _ensure_layout(cmds, name: str) -> Optional[str]:
    """Stellt sicher, dass das Shelf-Layout existiert. Gibt den Shelf-Namen zurück oder None."""
    name = _normalize_name(name)
    if not cmds.shelfTabLayout("ShelfLayout", q=True, ex=True):
        return None
    if not cmds.shelfLayout(name, q=True, ex=True):
        try:
            cmds.shelfLayout(name, p="ShelfLayout")
        except Exception:
            return None
    return name


def _iter_shelf_buttons(cmds, name: str) -> Iterable[str]:
    """Iterator über die UI-Objekte (Button-Names) eines Shelfs."""
    name = _normalize_name(name)
    try:
        children = cmds.shelfLayout(name, q=True, ca=True) or []
    except Exception:
        children = []
    for ch in children:
        try:
            if cmds.objectTypeUI(ch) == "shelfButton":
                yield ch
        except Exception:
            continue


def list_buttons(cmds, name: str) -> List[Tuple[str, str, str, str]]:
    """Für Debug: Liste (uiName, ann, label, command)."""
    cmds = _ensure_cmds(cmds)
    name = _normalize_name(name)
    out = []
    for ch in _iter_shelf_buttons(cmds, name):
        try:
            ann = cmds.shelfButton(ch, q=True, ann=True) or ""
        except Exception:
            ann = ""
        try:
            lbl = cmds.shelfButton(ch, q=True, l=True) or ""
        except Exception:
            lbl = ""
        try:
            cmd = cmds.shelfButton(ch, q=True, c=True) or ""
        except Exception:
            cmd = ""
        out.append((ch, ann, lbl, cmd))
    return out


# -----------------------------------------------------------------------------
# Button-Helfer
# -----------------------------------------------------------------------------

def _is_wrapper_file(path: str) -> bool:
    try:
        with open(path, "r", encoding="utf-8", errors="ignore") as f:
            head = f.read(len(_WRAPPER_HEADER))
        return head == _WRAPPER_HEADER
    except Exception:
        return False

def _resolve_bootstrap_icon(module_key: str, icon: str) -> str:
    if not icon:
        return ""
    if os.path.isabs(icon):
        return icon
    if R and hasattr(R, "full_icon_path"):
        path = R.full_icon_path(icon)
        if os.path.isabs(path) and os.path.exists(path):
            return path
    base_candidates = []
    if module_key == "hub":
        base_candidates.append(("hub", "icons", icon))
    if module_key in ("vd", "vertex_distributor"):
        base_candidates.append(("tools", "vertex_distributor", "icons", icon))
    base_candidates += [
        ("assets", "icons", icon),
        ("icons", icon),
    ]
    pkg_root = os.path.dirname(os.path.dirname(__file__))  # .../pjots_zen_tools
    for parts in base_candidates:
        cand = os.path.join(pkg_root, *parts)
        if os.path.exists(cand):
            return cand
    return icon


def _add_button(
    cmds,
    name: str,
    *,
    label: str,
    annot: str,
    command: str,
    icon: str = "",
    image_overlay_label: Optional[str] = None,
) -> Optional[str]:
    name = _normalize_name(name)
    if not _ensure_layout(cmds, name):
        return None
    kw = dict(p=name, l=label, ann=annot, stp="python", c=command)

    if icon:
        try:
            # absolut auflösen – identisch zum Hub-Verhalten
            from pjots_zen_tools import resources as R
            icon = R.full_icon_path(icon) or icon
        except Exception:
            pass
        kw["i"] = icon

    if image_overlay_label:
        kw["imageOverlayLabel"] = image_overlay_label[:4]
    try:
        return cmds.shelfButton(**kw)
    except Exception:
        return None



def remove_module_button(cmds, name: str, module_id: str) -> int:
    """
    Entfernt Buttons mit ann == "zen:<module_id>".
    Returns: Anzahl entfernter Buttons.
    """
    cmds = _ensure_cmds(cmds)
    name = _normalize_name(name)
    removed = 0
    for ch in list(_iter_shelf_buttons(cmds, name)):
        try:
            ann = cmds.shelfButton(ch, q=True, ann=True) or ""
        except Exception:
            ann = ""
        if ann == f"zen:{module_id}":
            try:
                cmds.deleteUI(ch)
                removed += 1
            except Exception:
                pass
    return removed


# -----------------------------------------------------------------------------
# Persistenz via Wrapper (kein mel.saveShelf!)
# -----------------------------------------------------------------------------
def _cleanup_double_ext(cmds, name: str) -> None:
    """Entfernt versehentlich erzeugte shelf_<name>.mel.mel."""
    try:
        n = _normalize_name(name)
        bad = os.path.join(_shelves_dir(_ensure_cmds(cmds)), f"shelf_{n}.mel.mel")
        if os.path.isfile(bad):
            os.remove(bad)
    except Exception:
        pass



_WRAPPER_HEADER = "// PZT shelf wrapper – auto-generated\n"


def _wrapper_proc_name(name: str) -> str:
    n = _normalize_name(name)
    return f"shelf_{n}"


def ensure_wrapper_file(cmds=None, name: str = "PjotsZenTools",
                        bootstrap: Iterable[str] = ("hub", "vd")) -> str:
    """
    Schreibt nur den MINIMAL‑Wrapper, und zwar nur dann,
    wenn keine echte Shelf‑Datei existiert (oder bereits unser Wrapper).
    """
    cmds = _ensure_cmds(cmds)
    name_n = _normalize_name(name)
    proc = _wrapper_proc_name(name_n)
    shelves_dir = _shelves_dir(cmds)
    os.makedirs(shelves_dir, exist_ok=True)
    path = _shelf_file_path(cmds, name_n)  # .../shelf_<name>.mel

    # echte (nicht‑Wrapper) Datei nicht überschreiben
    if os.path.isfile(path) and not _is_wrapper_file(path):
        _cleanup_double_ext(cmds, name_n)
        return path

    # Python‑Aufruf
    bs = ",".join(['\\"%s\\"' % b for b in (bootstrap or ())])
    py_call = (
        'python("import pjots_zen_tools.core.shelves as _s; '
        f'_s.build_or_sync(name=\\"{name_n}\\", bootstrap=[{bs}])");'
    )

    mel_lines = [
        _WRAPPER_HEADER,
        f"global proc {proc}() {{",
        "    global string $gShelfTopLevel;",
        f'    string $shelfName = "{name_n}";',
        "    if (!`shelfLayout -exists $shelfName`) {",
        "        shelfLayout $shelfName;",
        "    }",
        "    setParent $shelfName;",
        "    // Dynamischer Python-Builder",
        f"    {py_call}",
        "}",
        "",
    ]
    new = "\n".join(mel_lines)

    try:
        if os.path.isfile(path) and _is_wrapper_file(path):
            with open(path, "r", encoding="utf-8", errors="ignore") as f:
                old = f.read()
            if old == new:
                _cleanup_double_ext(cmds, name_n)
                return path
    except Exception:
        pass

    with open(path, "w", encoding="utf-8", errors="ignore") as f:
        f.write(new)

    _cleanup_double_ext(cmds, name_n)
    return path


def save_shelf(cmds=None, name: str = "PjotsZenTools") -> str:
    p = ensure_wrapper_file(cmds=cmds, name=name, bootstrap=("hub", "vd"))
    _cleanup_double_ext(cmds, name)
    return p

# -----------------------------------------------------------------------------
# Sync-Logik mit Registry
# -----------------------------------------------------------------------------
def _installed_module_ids(registry: Dict[str, Dict]) -> List[str]:
    """
    Welche module_ids gelten als installiert?
    Erwartet Registry-Format von hub.registry.get_tools():
      {module_id: {installed: bool, ...}}
    """
    if not registry:
        return []
    out = []
    for mid, meta in registry.items():
        if bool(meta.get("installed")):
            out.append(mid)
    return out


def sync_shelf_with_registry(cmds, name: str, registry: Dict[str, Dict]) -> int:
    """
    Entfernt verwaiste `zen:*`-Buttons, deren module_id nicht (mehr) installiert ist.
    Schützt den Hub-Eintrag.
    Returns: Anzahl entfernter Buttons.
    """
    cmds = _ensure_cmds(cmds)
    name = _normalize_name(name)
    installed = set(_installed_module_ids(registry))
    removed = 0
    for ch in list(_iter_shelf_buttons(cmds, name)):
        try:
            ann = cmds.shelfButton(ch, q=True, ann=True) or ""
        except Exception:
            ann = ""
        if ann.startswith("zen:"):
            mod = ann.split(":", 1)[-1]
            if mod and mod not in installed and mod not in ("hub",):
                try:
                    cmds.deleteUI(ch)
                    removed += 1
                except Exception:
                    pass
    return removed


# -----------------------------------------------------------------------------
# Helper (oberhalb der Bootstrap‑Sektion)
# -----------------------------------------------------------------------------
def _cmd_registry_first(module_id: str, fallback_entry: str) -> str:
    """
    Liefert Python-Code (als String) für Shelf-Buttons:
      1) Registry → launcher() falls vorhanden
      2) Fallback → importlib auf 'fallback_entry' (e.g. 'pkg.mod:func')
    """

    mod, func = fallback_entry.split(":", 1)
    return (
        "import importlib\n"
        "import pjots_zen_tools.hub.registry as _reg\n"
        f"_mid='{module_id}'\n"
        "_m=_reg.get_tools().get(_mid)\n"
        "_l=_m.get('launcher') if _m else None\n"
        "if callable(_l):\n"
        "    _l()\n"
        "else:\n"
        f"    _m = importlib.import_module('{mod}')\n"
        f"    getattr(_m, '{func}')()\n"
    )


# -----------------------------------------------------------------------------
# Bootstrap-Buttons (Hub & Vertex Distributor)
# -----------------------------------------------------------------------------
_HUB = {
    "module_id": "hub",
    "label": "ZenTools Hub",
    "annot": "zen:hub",
    "icon": "pjots_zen_tools_hub.png",
    "fallback_entry": "pjots_zen_tools.hub.hub_ui:show",
}

_VD = {
    "module_id": "vertex_distributor",
    "label": "Vertex Distributor",
    "annot": "zen:vertex_distributor",
    "icon": "vertex_distributor.png",
    "fallback_entry": "pjots_zen_tools.tools.vertex_distributor.launch:show",
}


def _bootstrap_buttons(cmds, name: str, which: Iterable[str]) -> None:
    """
    Legt gewünschte Bootstrap-Buttons an (falls nicht vorhanden).
    `which` enthält z. B. ['hub','vd'].
    """
    cmds = _ensure_cmds(cmds)
    name = _normalize_name(name)
    want = []
    for key in (which or []):
        k = (key or "").strip().lower()
        if k == "hub":
            want.append(_HUB)
        elif k in ("vd", "vertex_distributor"):
            want.append(_VD)

    if not want:
        return

    # Existierende Annotations sammeln
    existing_anns = set()
    for ch in _iter_shelf_buttons(cmds, name):
        try:
            existing_anns.add(cmds.shelfButton(ch, q=True, ann=True) or "")
        except Exception:
            pass

    # Reihenfolge: Hub zuerst, dann VD
    for spec in want:
        if spec["annot"] in existing_anns:
            continue
        cmd = _cmd_registry_first(spec["module_id"], spec["fallback_entry"])  # dynamischer Command
        key_for_icon = "vd" if spec["module_id"] in ("vertex_distributor",) else "hub"
        _add_button(cmds, name, label=spec["label"], annot=spec["annot"], command=cmd, icon=(R.full_icon_path(spec["icon"]) if R else spec["icon"]),
            image_overlay_label=spec["label"],
        )


# -----------------------------------------------------------------------------
# Hauptfunktion: Build oder Sync
# -----------------------------------------------------------------------------
def build_or_sync(cmds=None, name: str = "PjotsZenTools",
                  bootstrap: Optional[Iterable[str]] = None) -> None:
    """
    Baut (oder aktualisiert) das Shelf:
      1) Shelf-Layout sicherstellen
      2) Optional Bootstrap-Buttons anlegen (z. B. ['hub','vd'])
      3) Registry laden und verwaiste zen:* Buttons entfernen (Bootstrap geschützt)
      4) Wrapper-Datei sicherstellen (Persistenz)

    Aufrufbar aus MEL (über Installer) via:
      python("import pjots_zen_tools.core.shelves as _s; _s.build_or_sync(name=\"PjotsZenTools\", bootstrap=[\"hub\", \"vd\"]) ");
    """
    cmds = _ensure_cmds(cmds)
    name = _normalize_name(name)

    # 1) Layout sicherstellen
    if not _ensure_layout(cmds, name):
        return

    # 2) Bootstrap (optional) – Buttons anlegen, falls nicht vorhanden
    try:
        if bootstrap:
            _bootstrap_buttons(cmds, name, bootstrap)
    except Exception:
        pass

    # 3) Registry lesen
    try:
        from pjots_zen_tools.hub.registry import get_tools  # lazy import
        registry = get_tools() or {}
    except Exception:
        registry = {}

    # 3b) Bootstrap-Module beim Aufräumen schützen
    protect = set()
    for key in (bootstrap or ()):
        k = (key or "").strip().lower()
        if k == "hub":
            protect.add("hub")
        elif k in ("vd", "vertex_distributor"):
            protect.add("vertex_distributor")

    # 4) Verwaiste zen:* Buttons entfernen – aber Bootstrap geschützt lassen
    try:
        installed = set(_installed_module_ids(registry))
        for ch in list(_iter_shelf_buttons(cmds, name)):
            try:
                ann = cmds.shelfButton(ch, q=True, ann=True) or ""
            except Exception:
                ann = ""
            if not ann.startswith("zen:"):
                continue
            mod = ann.split(":", 1)[-1]
            if mod in protect:
                continue
            if mod and mod not in installed:
                try:
                    cmds.deleteUI(ch)
                except Exception:
                    pass
    except Exception:
        pass

    # 5) Wrapper sicherstellen (Persistenz)
    try:
        ensure_wrapper_file(cmds=cmds, name=name, bootstrap=("hub", "vd"))
    except Exception:
        pass
