# src/pjots_zen_tools/hub/registry.py
from __future__ import annotations

from typing import Dict, Callable, Any, Optional
import threading

__all__ = [
    "register_tool",
    "unregister_tool",
    "update_tool",
    "set_installed",
    "is_registered",
    "get_tools",
    "clear_registry",
]

# Interner Speicher
_REGISTRY: Dict[str, Dict[str, Any]] = {}
_LOCK = threading.RLock()


def _normalize_meta(
    module_id: str,
    title: str,
    launcher: Optional[Callable],
    icon: str,
    version: str,
    tags: Optional[list],
    description: str,
    path: Optional[str] = None,
    installed: Optional[bool] = None,
) -> Dict[str, Any]:
    """
    Vereinheitlicht Metadaten. 'installed' defaultet auf True für registrierte Tools.
    """
    return {
        "module_id": module_id,
        "title": title or module_id,
        "launcher": launcher,
        "icon": icon or "",
        "version": version or "0.0.0",
        "tags": list(tags or []),
        "description": description or "",
        # optionale Felder, die u. a. der Hub nutzt/weiterreicht:
        "path": path or "",
        "installed": True if installed is None else bool(installed),
    }


def register_tool(
    module_id: str,              # z.B. "vertex_distributor"
    title: str,
    launcher: Callable | None,
    icon: str = "",
    version: str = "1.0.0",
    tags: Optional[list] = None,
    description: str = "",
    *,
    path: Optional[str] = None,
    installed: Optional[bool] = None,
) -> None:
    """
    Registriert/aktualisiert ein Tool im In‑Memory‑Katalog.
    Backwards‑compat: gleiche Signatur wie vorher plus optionale Keywords.
    """
    meta = _normalize_meta(
        module_id=module_id,
        title=title,
        launcher=launcher,
        icon=icon,
        version=version,
        tags=tags,
        description=description,
        path=path,
        installed=installed,
    )
    with _LOCK:
        _REGISTRY[module_id] = meta


def unregister_tool(module_id: str) -> bool:
    """
    Entfernt ein Tool vollständig aus der Registry.
    Returns: True, wenn vorhanden war und entfernt wurde.
    """
    with _LOCK:
        return _REGISTRY.pop(module_id, None) is not None


def set_installed(module_id: str, installed: bool) -> bool:
    """
    Markiert ein bereits registriertes Tool als installiert/nicht installiert,
    ohne Metadaten zu verlieren. Praktisch für sofortiges UI‑Feedback nach Uninstall.
    Returns: True bei Erfolg.
    """
    with _LOCK:
        meta = _REGISTRY.get(module_id)
        if not meta:
            return False
        meta["installed"] = bool(installed)
        # Launcher optional kappen, wenn nicht installiert (härterer Schutz)
        if not installed:
            meta["launcher"] = None
        return True


def update_tool(module_id: str, **fields: Any) -> bool:
    """
    Aktualisiert einzelne Felder eines registrierten Tools.
    Erlaubt u. a. 'version', 'icon', 'launcher', 'tags', 'description', 'path', 'installed'.
    Returns: True bei Erfolg.
    """
    with _LOCK:
        meta = _REGISTRY.get(module_id)
        if not meta:
            return False
        # Nur bekannte Keys zulassen (sanitizing)
        allowed = {
            "title", "launcher", "icon", "version", "tags",
            "description", "path", "installed",
        }
        for k, v in list(fields.items()):
            if k not in allowed:
                continue
            if k == "tags" and v is not None:
                meta[k] = list(v)
            elif k == "installed":
                meta[k] = bool(v)
                if not meta[k]:
                    meta["launcher"] = None
            else:
                meta[k] = v
        return True


def is_registered(module_id: str) -> bool:
    with _LOCK:
        return module_id in _REGISTRY


def get_tools(*, copy: bool = True) -> Dict[str, Dict[str, Any]]:
    """
    Liefert die Registry. Standard: Kopie (Side‑Effects vermeiden).
    Für Performance‑kritische Call‑Sites kann copy=False genutzt werden.
    """
    with _LOCK:
        return dict(_REGISTRY) if copy else _REGISTRY


def clear_registry() -> None:
    """Leert die Registry (für Tests/Reloads)."""
    with _LOCK:
        _REGISTRY.clear()
