# pjots_zen_tools/core/qt_compat.py
# Vereinheitlicht PySide2/PySide6 + shiboken2/shiboken6 und nützliche Helfer.
from __future__ import annotations

# --- Qt laden: bevorzugt PySide2 (Maya 2022–2025), Fallback PySide6 ---
try:
    from PySide2 import QtCore, QtGui, QtWidgets  # type: ignore
    API = "PySide2"
    IS_QT6 = False
except Exception:
    from PySide6 import QtCore, QtGui, QtWidgets  # type: ignore
    API = "PySide6"
    IS_QT6 = True

# --- shiboken optional laden (2 oder 6), kann auch fehlen ---
_shiboken = None
try:
    import shiboken2 as _shiboken  # type: ignore
except Exception:
    try:
        import shiboken6 as _shiboken  # type: ignore
    except Exception:
        # Manchmal existiert "shiboken" ohne Suffix – als letzter Versuch:
        try:
            import shiboken as _shiboken  # type: ignore
        except Exception:
            _shiboken = None  # im Standalone ggf. nicht vorhanden

def shiboken():
    """Gibt das shiboken-Modul zurück (oder None)."""
    return _shiboken

def wrap_instance(ptr: int, base_type):
    """
    Qt-Wrapper für C++-Pointer → Python-Objekt (sicher für shiboken2/6).
    Gibt None zurück, wenn shiboken nicht verfügbar ist oder wrapping fehlschlägt.
    """
    if _shiboken is None or ptr is None:
        return None
    try:
        return _shiboken.wrapInstance(int(ptr), base_type)
    except Exception:
        # Manche Builds wollen explizit int/long – Retry wurde bereits mit int() gemacht.
        return None

# QAction-Kompatibilität (PySide6 verschob QAction nach QtGui)
try:
    QAction = QtWidgets.QAction
except AttributeError:
    QAction = QtGui.QAction

# --- QApplication Helfer ---
def get_app():
    """Existierende QApplication (oder None)."""
    try:
        return QtWidgets.QApplication.instance()
    except Exception:
        return None

def ensure_app(app_name: str = "PjotsZenTools"):
    """
    Erstellt eine QApplication, falls keine existiert (nur für Standalone-Tests).
    In Maya normalerweise NICHT verwenden.
    """
    app = get_app()
    if app:
        return app
    try:
        app = QtWidgets.QApplication([app_name])
        if IS_QT6:
            # Sanfte DPI-Policy für moderne Monitore
            QtWidgets.QApplication.setHighDpiScaleFactorRoundingPolicy(
                QtCore.Qt.HighDpiScaleFactorRoundingPolicy.PassThrough
            )
        return app
    except Exception:
        return None

__all__ = [
    "QtCore", "QtGui", "QtWidgets",
    "API", "IS_QT6", "QAction",
    "shiboken", "wrap_instance",
    "get_app", "ensure_app",
]
